import { hasFunction, IAppCounterService, IParam } from "@core";

/**
 * @description 应用计数器服务
 * @export
 * @class AppCounterService
 * @implements {IAppCounterService}
 */
export class AppCounterService implements IAppCounterService {

    /**
     * @description 定时器延时时间
     * @protected
     * @type {number}
     * @memberof AppCounterService
     */
    protected delayTime: number = 6000;

    /**
     * @description 计数器对象
     * @protected
     * @type {*}
     * @memberof AppCounterService
     */
    protected timer: any = null;

    /**
     * @description 获取数据行为
     * @protected
     * @type {string}
     * @memberof AppCounterService
     */
    protected getAction: string = '';

    /**
     * @description 计数器唯一标识
     * @type {string}
     * @memberof AppCounterService
     */
    public id: string = '';

    /**
     * @description 计数器实体标识
     * @private
     * @type {string}
     * @memberof AppCounterService
     */
    private deCodeName: string = '';

    /**
     * @description 实体服务
     * @private
     * @type {*}
     * @memberof AppCounterService
     */
    private entityService: any;

    /**
     * @description 计数器数据
     * @private
     * @type {*}
     * @memberof AppCounterService
     */
    private counterData: any;

    /**
     * @description 获取计数器数据
     * @readonly
     * @memberof AppCounterService
     */
    get data(): any {
        return this.counterData;
    }

    /**
     * Creates an instance of AppCounterService.
     * @param {IParam} counterRef 计数器
     * @param {IParam} [context={}] 应用上下文
     * @param {IParam} [viewParams={}] 视图参数
     * @memberof AppCounterService
     */
    constructor (counterRef: IParam, private context: IParam = {}, private viewParams: IParam = {}) {
        this.delayTime = counterRef.timer;
        this.getAction = counterRef.getAction;
        this.deCodeName = counterRef.deCodeName;
        this.id = counterRef.id;
    }

    /**
     * @description 执行计数器
     * @memberof AppCounterService
     */
    async execute() {
        await this.getCounterData();
        this.destory();
        this.timer = setInterval(() => {
            this.getCounterData();
        }, this.delayTime);
    }

    /**
     * @description 获取计数器数据
     * @private
     * @return {*} 
     * @memberof AppCounterService
     */
    private async getCounterData() {
        //  获取实体服务
        if (this.deCodeName && !this.entityService) {
            this.entityService = await App.getDataService(this.deCodeName, this.context);
        }
        if (!this.entityService || !this.getAction) {
            return;
        }
        if (hasFunction(this.entityService, this.getAction)) {
            try {
                const response: any = await this.entityService[this.getAction](this.context, this.viewParams);
                if (response && response.status === 200 && response.data) {
                    this.counterData = response.data;
                }
            } catch (error: any) {
                console.error(error);
            }
        }
    }

    /**
     * @description 销毁定时器
     * @memberof AppCounterService
     */
    destory() {
        if (this.timer) {
            clearInterval(this.timer);
        }
    }
}